#!/usr/bin/env bash
set -euo pipefail

PKG_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
TARGET_DIR="/root/cybertoolsadmin"
LICENSE_DIR="/root/.cybertoolsadmin"
LICENSE_FILE="$LICENSE_DIR/license.key"
LOG="$TARGET_DIR/cybertools-admin.log"
BASHRC="/root/.bashrc"

GREEN="\033[1;32m"; YELLOW="\033[1;33m"; RED="\033[1;31m"; CYAN="\033[1;36m"; RESET="\033[0m"

echo -e "${CYAN}=== CyberTools Admin – Installer ===${RESET}"

# 0) Root check
if [[ "$EUID" -ne 0 ]]; then
  echo -e "${RED}Please run as root.${RESET}"
  exit 1
fi

# 1) Create target folders
mkdir -p "$TARGET_DIR" "$LICENSE_DIR"
touch "$LOG"
chmod 600 "$LICENSE_DIR"
echo -e "${GREEN}✓ Folders ready${RESET}"

# 2) Copy binaries
install -m 0755 "$PKG_DIR/admin-menu"    "$TARGET_DIR/admin-menu"
install -m 0755 "$PKG_DIR/admin-backup"  "$TARGET_DIR/admin-backup"
install -m 0755 "$PKG_DIR/admin-restore" "$TARGET_DIR/admin-restore"
echo -e "${GREEN}✓ Binaries installed to $TARGET_DIR${RESET}"

# 3) Copy manifests + sample config (don't overwrite live config.conf if exists)
[[ -f "$PKG_DIR/dependencies.txt" ]]   && install -m 0644 "$PKG_DIR/dependencies.txt"   "$TARGET_DIR/dependencies.txt"
[[ -f "$PKG_DIR/system-deps.txt" ]]    && install -m 0644 "$PKG_DIR/system-deps.txt"    "$TARGET_DIR/system-deps.txt"
[[ -f "$TARGET_DIR/config.conf" ]]     || install -m 0644 "$PKG_DIR/config.sample.conf" "$TARGET_DIR/config.conf"
echo -e "${GREEN}✓ Manifests + config ready${RESET}"

# 4) License setup
echo
echo -e "${CYAN}License Setup${RESET}"
read -rp "Provide license passphrase now (or leave blank to skip): " LIC
if [[ -n "${LIC:-}" ]]; then
  echo "$LIC" > "$LICENSE_FILE"
  chmod 600 "$LICENSE_FILE"
  echo -e "${GREEN}✓ License saved to $LICENSE_FILE${RESET}"
else
  echo -e "${YELLOW}Skipped license entry. You can create $LICENSE_FILE later.${RESET}"
fi

# 5) Autostart toggle
echo
echo -e "${CYAN}Autostart on root login?${RESET}"
read -rp "Enable autostart now? [Y/n]: " ans
ans="${ans:-Y}"
AUTOSTART_TAG="# CyberToolsAdmin Autostart"
if [[ "$ans" =~ ^[Yy]$ ]]; then
  # Remove any old block then append fresh block for /root/cybertoolsadmin/admin-menu
  sed -i "/$AUTOSTART_TAG/,+3d" "$BASHRC" || true
  cat >> "$BASHRC" <<'EOL'

# CyberToolsAdmin Autostart
if [[ $- == *i* ]] && [ -x /root/cybertoolsadmin/admin-menu ]; then
  /root/cybertoolsadmin/admin-menu
fi
EOL
  echo -e "${GREEN}✓ Autostart ENABLED (login as root to see the menu)${RESET}"
else
  sed -i "/$AUTOSTART_TAG/,+3d" "$BASHRC" || true
  echo -e "${YELLOW}Autostart left DISABLED${RESET}"
fi

# 6) Optional: install dependencies
echo
if [[ -f "$TARGET_DIR/dependencies.txt" ]]; then
  echo -e "${CYAN}Install Python dependencies from dependencies.txt?${RESET}"
  read -rp "Proceed with pip3 install -r (each line)? [y/N]: " p
  if [[ "$p" =~ ^[Yy]$ ]]; then
    while IFS= read -r raw || [[ -n "$raw" ]]; do
      line="$(echo "$raw" | sed 's/#.*$//' | xargs)"
      [[ -z "$line" ]] && continue
      echo -e "${YELLOW}Installing: $line${RESET}"
      pip3 install -U "$line"
    done < "$TARGET_DIR/dependencies.txt"
    echo -e "${GREEN}✓ Python deps done${RESET}"
  fi
fi

if [[ -f "$TARGET_DIR/system-deps.txt" ]]; then
  echo -e "${CYAN}Install system packages from system-deps.txt (apt)?${RESET}"
  read -rp "Proceed with apt-get install? [y/N]: " q
  if [[ "$q" =~ ^[Yy]$ ]]; then
    apt-get update
    xargs -a "$TARGET_DIR/system-deps.txt" -r apt-get install -y
    echo -e "${GREEN}✓ System deps done${RESET}"
  fi
fi

echo
echo -e "${GREEN}Installation complete.${RESET}"
echo -e "Re-login as root to see ${CYAN}CyberTools Admin Menu${RESET}."
